<?php
/**
 * The public-facing functionality of the plugin.
 */
class Licenzer_Public {

    /**
     * The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     */
    public function enqueue_styles() {
        wp_enqueue_style(
            $this->plugin_name,
            plugin_dir_url(__FILE__) . 'css/licenzer-public.css',
            array(),
            $this->version,
            'all'
        );
    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     */
    public function enqueue_scripts() {
        wp_enqueue_script(
            $this->plugin_name,
            plugin_dir_url(__FILE__) . 'js/licenzer-public.js',
            array('jquery'),
            $this->version,
            false
        );

        wp_localize_script($this->plugin_name, 'licenzerAjax', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('licenzer_public_nonce'),
        ));
    }

    /**
     * Register shortcodes
     */
    public function register_shortcodes() {
        add_shortcode('licenzer_my_licenses', array($this, 'display_my_licenses'));
        add_shortcode('licenzer_change_ip', array($this, 'display_change_ip_form'));
        
        // Register AJAX handlers
        add_action('wp_ajax_licenzer_change_ip', array($this, 'handle_change_ip'));
    }

    /**
     * Display user licenses shortcode
     */
    public function display_my_licenses($atts) {
        if (!is_user_logged_in()) {
            return '<p>' . __('Please log in to view your licenses.', 'licenzer') . '</p>';
        }

        $user_id = get_current_user_id();
        global $wpdb;
        $table_name = $wpdb->prefix . 'licenzer_licenses';
        
        $licenses = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE user_id = %d ORDER BY created_at DESC",
            $user_id
        ));

        ob_start();
        include plugin_dir_path(__FILE__) . 'partials/my-licenses.php';
        return ob_get_clean();
    }

    /**
     * Display change IP form shortcode
     */
    public function display_change_ip_form($atts) {
        if (!is_user_logged_in()) {
            return '<p>' . __('Please log in to change IP address.', 'licenzer') . '</p>';
        }

        $atts = shortcode_atts(array(
            'license_key' => '',
        ), $atts);

        ob_start();
        include plugin_dir_path(__FILE__) . 'partials/change-ip-form.php';
        return ob_get_clean();
    }

    /**
     * Handle change IP AJAX request
     */
    public function handle_change_ip() {
        check_ajax_referer('licenzer_public_nonce', 'nonce');

        if (!is_user_logged_in()) {
            wp_send_json_error(array('message' => __('Please log in.', 'licenzer')));
            return;
        }

        if (get_option('licenzer_allow_ip_change') != 'yes') {
            wp_send_json_error(array('message' => __('IP change is not allowed.', 'licenzer')));
            return;
        }

        $license_key = sanitize_text_field($_POST['license_key'] ?? '');
        $new_ip = sanitize_text_field($_POST['new_ip'] ?? '');

        if (empty($license_key) || empty($new_ip)) {
            wp_send_json_error(array('message' => __('License key and IP address are required.', 'licenzer')));
            return;
        }

        if (!filter_var($new_ip, FILTER_VALIDATE_IP)) {
            wp_send_json_error(array('message' => __('Invalid IP address format.', 'licenzer')));
            return;
        }

        // Get license info from database
        global $wpdb;
        $table_name = $wpdb->prefix . 'licenzer_licenses';
        $license = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE license_key = %s AND user_id = %d",
            $license_key,
            get_current_user_id()
        ));

        if (!$license) {
            wp_send_json_error(array('message' => __('License not found.', 'licenzer')));
            return;
        }

        // Check IP change limit
        $max_changes = intval(get_option('licenzer_max_ip_changes', '0'));
        if ($max_changes > 0 && $license->ip_changes >= $max_changes) {
            wp_send_json_error(array('message' => __('Maximum IP changes limit reached.', 'licenzer')));
            return;
        }

        // Call API to change IP
        $api = new Licenzer_API();
        $result = $api->changeIp($license->ip_address, $new_ip, $license->software_key);

        if (isset($result['status']) && $result['status'] == 'success') {
            // Update database
            $wpdb->update(
                $table_name,
                array(
                    'ip_address' => $new_ip,
                    'ip_changes' => $license->ip_changes + 1,
                ),
                array('id' => $license->id)
            );

            wp_send_json_success(array('message' => __('IP address changed successfully.', 'licenzer')));
        } else {
            wp_send_json_error(array('message' => $result['message'] ?? __('Failed to change IP address.', 'licenzer')));
        }
    }
}

