<?php

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

use WHMCS\Database\Capsule;

/**
 * LicenZer Server Module
 * 
 * @see https://developers.whmcs.com/provisioning-modules/
 */

function licenzer_MetaData()
{
    return array(
        'DisplayName' => 'LicenZer License Manager',
        'APIVersion' => '1.1',
        'RequiresServer' => false,
        'DefaultNonSSLPort' => '80',
        'DefaultSSLPort' => '443',
        'ServiceSingleSignOnLabel' => 'Login to Panel',
        'AdminSingleSignOnLabel' => 'Login to Admin Panel',
    );
}

function licenzer_ConfigOptions()
{
    return array(
        'Reseller Token' => array(
            'Type' => 'text',
            'Size' => '50',
            'Description' => 'Enter your LicenZer reseller token',
        ),
        'API URL' => array(
            'Type' => 'text',
            'Size' => '100',
            'Default' => 'https://licenzer.pro',
            'Description' => 'LicenZer API base URL',
        ),
        'Software Key' => array(
            'Type' => 'text',
            'Size' => '50',
            'Description' => 'Software key (e.g., cpanel, cloudlinux)',
        ),
        'Allow IP Change' => array(
            'Type' => 'yesno',
            'Description' => 'Allow clients to change license IP',
        ),
        'Max IP Changes' => array(
            'Type' => 'text',
            'Size' => '10',
            'Default' => '0',
            'Description' => 'Maximum IP change attempts (0 = unlimited)',
        ),
        'Enable Discount Licenses' => array(
            'Type' => 'yesno',
            'Description' => 'Enable discount licenses group',
        ),
    );
}

function licenzer_CreateAccount($params)
{
    try {
        $api = new LicenZerAPI($params);
        
        // Get IP from custom field or service
        $ip = $params['customfields']['IP Address'] ?? $params['model']['dedicatedip'] ?? $params['serverip'] ?? '';
        
        if (empty($ip)) {
            return 'IP address is required. Please set IP address in service custom fields or server IP.';
        }
        
        // Get duration (default 1 month)
        $month = $params['configoptions']['Duration'] ?? $params['model']['billingcycle'] ?? 1;
        // Convert billing cycle to months
        if (is_string($month)) {
            $monthMap = ['monthly' => 1, 'quarterly' => 3, 'semiannually' => 6, 'annually' => 12, 'biennially' => 24, 'triennially' => 36];
            $month = $monthMap[strtolower($month)] ?? 1;
        }
        
        // Register license
        $result = $api->registerLicense($ip, $month);
        
        if ($result['success']) {
            $licenseKey = $result['license_key'] ?? '';
            
            // Store license key in service notes
            Capsule::table('tblhosting')
                ->where('id', $params['serviceid'])
                ->update(['notes' => json_encode([
                    'license_key' => $licenseKey,
                    'ip_changes' => 0,
                    'max_ip_changes' => $params['configoption4'] ?? 0,
                    'ip' => $ip,
                ])]);
            
            // Store license key in custom field if exists
            $customField = Capsule::table('tblcustomfields')
                ->where('type', 'product')
                ->where('relid', $params['pid'])
                ->where('fieldname', 'LIKE', '%License Key%')
                ->first();
            
            if ($customField) {
                Capsule::table('tblcustomfieldsvalues')
                    ->updateOrInsert(
                        ['fieldid' => $customField->id, 'relid' => $params['serviceid']],
                        ['value' => $licenseKey]
                    );
            }
            
            return 'success';
        } else {
            return $result['message'] ?? 'Failed to create license';
        }
    } catch (Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
}

function licenzer_SuspendAccount($params)
{
    try {
        $api = new LicenZerAPI($params);
        $licenseKey = licenzer_GetLicenseKey($params);
        
        if (empty($licenseKey)) {
            return 'License key not found';
        }
        
        $result = $api->deactivateLicense($licenseKey);
        
        if ($result['success']) {
            return 'success';
        } else {
            return $result['message'] ?? 'Failed to suspend license';
        }
    } catch (Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
}

function licenzer_UnsuspendAccount($params)
{
    try {
        $api = new LicenZerAPI($params);
        $licenseKey = licenzer_GetLicenseKey($params);
        
        if (empty($licenseKey)) {
            return 'License key not found';
        }
        
        $result = $api->activateLicense($licenseKey);
        
        if ($result['success']) {
            return 'success';
        } else {
            return $result['message'] ?? 'Failed to unsuspend license';
        }
    } catch (Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
}

function licenzer_TerminateAccount($params)
{
    try {
        $api = new LicenZerAPI($params);
        $licenseKey = licenzer_GetLicenseKey($params);
        
        if (empty($licenseKey)) {
            return 'License key not found';
        }
        
        $result = $api->deleteLicense($licenseKey);
        
        if ($result['success']) {
            return 'success';
        } else {
            return $result['message'] ?? 'Failed to terminate license';
        }
    } catch (Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
}

function licenzer_ChangePassword($params)
{
    // Not applicable for licenses
    return 'success';
}

function licenzer_ChangePackage($params)
{
    // Handle package change if needed
    return 'success';
}

function licenzer_AdminCustomButtonArray()
{
    // Admin functions moved to Addon module
    // Keep only essential service management buttons
    return array(
        "View License Info" => "ViewLicenseInfo",
    );
}

function licenzer_ClientArea($params)
{
    try {
        $api = new LicenZerAPI($params);
        $licenseKey = licenzer_GetLicenseKey($params);
        
        // Handle IP change request
        if (isset($_POST['changeip']) && $_POST['changeip'] == '1') {
            return licenzer_HandleClientIpChange($params, $api, $licenseKey);
        }
        
        if (empty($licenseKey)) {
            return array(
                'templatefile' => 'client',
                'vars' => array(
                    'error' => 'License key not found',
                ),
            );
        }
        
        // Get license info
        $licenseInfo = $api->getLicenseInfo($licenseKey);
        
        // Get service notes
        $service = Capsule::table('tblhosting')
            ->where('id', $params['serviceid'])
            ->first();
        
        $notes = json_decode($service->notes ?? '{}', true);
        $ipChanges = $notes['ip_changes'] ?? 0;
        $maxIpChanges = $params['configoption4'] ?? 0;
        $allowIpChange = $params['configoption3'] ?? false;
        
        // Get installation command from license info
        $installCommand = '';
        if (isset($licenseInfo['cmd']) && !empty($licenseInfo['cmd'])) {
            $installCommand = $licenseInfo['cmd'];
        } else {
            $installCommand = $api->getInstallCommand($params['configoption3']);
        }
        
        return array(
            'templatefile' => 'client',
            'vars' => array(
                'license_key' => $licenseKey,
                'license_info' => $licenseInfo,
                'install_command' => $installCommand,
                'ip' => $licenseInfo['ip'] ?? '',
                'status' => $licenseInfo['status'] ?? 0,
                'expiry_date' => $licenseInfo['end_at'] ?? '',
                'allow_ip_change' => $allowIpChange,
                'ip_changes' => $ipChanges,
                'max_ip_changes' => $maxIpChanges,
                'can_change_ip' => $allowIpChange && ($maxIpChanges == 0 || $ipChanges < $maxIpChanges),
                'serviceid' => $params['serviceid'],
            ),
        );
    } catch (Exception $e) {
        return array(
            'templatefile' => 'client',
            'vars' => array(
                'error' => $e->getMessage(),
            ),
        );
    }
}

function licenzer_HandleClientIpChange($params, $api, $licenseKey)
{
    $newIp = $_POST['new_ip'] ?? '';
    
    if (empty($newIp) || !filter_var($newIp, FILTER_VALIDATE_IP)) {
        return array(
            'templatefile' => 'client',
            'vars' => array(
                'error' => 'Invalid IP address',
            ),
        );
    }
    
    // Check if IP change is allowed
    $allowIpChange = $params['configoption3'] ?? false;
    if (!$allowIpChange) {
        return array(
            'templatefile' => 'client',
            'vars' => array(
                'error' => 'IP change is not allowed for this service',
            ),
        );
    }
    
    // Get service notes
    $service = Capsule::table('tblhosting')
        ->where('id', $params['serviceid'])
        ->first();
    
    $notes = json_decode($service->notes ?? '{}', true);
    $ipChanges = $notes['ip_changes'] ?? 0;
    $maxIpChanges = $params['configoption4'] ?? 0;
    
    // Check if max IP changes reached
    if ($maxIpChanges > 0 && $ipChanges >= $maxIpChanges) {
        return array(
            'templatefile' => 'client',
            'vars' => array(
                'error' => 'Maximum IP changes reached',
            ),
        );
    }
    
    // Change IP
    $result = $api->changeIp($licenseKey, $newIp);
    
    if ($result['success']) {
        // Update IP change count
        $notes['ip_changes'] = $ipChanges + 1;
        Capsule::table('tblhosting')
            ->where('id', $params['serviceid'])
            ->update(['notes' => json_encode($notes)]);
        
        return array(
            'templatefile' => 'client',
            'vars' => array(
                'success' => 'IP address changed successfully',
                'license_key' => $licenseKey,
            ),
        );
    } else {
        return array(
            'templatefile' => 'client',
            'vars' => array(
                'error' => $result['message'] ?? 'Failed to change IP address',
            ),
        );
    }
}

function licenzer_AdminServicesTabFields($params)
{
    try {
        $api = new LicenZerAPI($params);
        $licenseKey = licenzer_GetLicenseKey($params);
        
        if (empty($licenseKey)) {
            return array();
        }
        
        $licenseInfo = $api->getLicenseInfo($licenseKey);
        
        return array(
            'License Key' => '<input type="text" class="form-control" value="' . htmlspecialchars($licenseKey) . '" readonly>',
            'IP Address' => '<input type="text" class="form-control" value="' . htmlspecialchars($licenseInfo['ip'] ?? '') . '" readonly>',
            'Status' => '<span class="label label-' . ($licenseInfo['status'] ? 'success' : 'danger') . '">' . ($licenseInfo['status'] ? 'Active' : 'Inactive') . '</span>',
            'Expiry Date' => $licenseInfo['end_at'] ?? 'N/A',
        );
    } catch (Exception $e) {
        return array(
            'Error' => $e->getMessage(),
        );
    }
}

function licenzer_AdminServicesTabFieldsSave($params)
{
    // Handle admin field saves if needed
    return array();
}

function licenzer_ViewLicenseInfo($params)
{
    try {
        $api = new LicenZerAPI($params);
        $licenseKey = licenzer_GetLicenseKey($params);
        
        if (empty($licenseKey)) {
            return 'License key not found';
        }
        
        $licenseInfo = $api->getLicenseInfo($licenseKey);
        
        $output = '<div class="alert alert-info">';
        $output .= '<h4>License Information</h4>';
        $output .= '<p><strong>License Key:</strong> ' . htmlspecialchars($licenseKey) . '</p>';
        $output .= '<p><strong>IP Address:</strong> ' . htmlspecialchars($licenseInfo['ip'] ?? '') . '</p>';
        $output .= '<p><strong>Status:</strong> ' . ($licenseInfo['status'] ? 'Active' : 'Inactive') . '</p>';
        $output .= '<p><strong>Expiry Date:</strong> ' . ($licenseInfo['end_at'] ?? 'N/A') . '</p>';
        $output .= '</div>';
        
        return $output;
    } catch (Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
}

function licenzer_ChangeIP($params)
{
    // Basic IP change - for advanced management use LicenZer Addon
    try {
        $api = new LicenZerAPI($params);
        $licenseKey = licenzer_GetLicenseKey($params);
        
        if (empty($licenseKey)) {
            return 'License key not found';
        }
        
        // Get new IP from POST
        $newIp = $_POST['new_ip'] ?? '';
        
        if (empty($newIp) || !filter_var($newIp, FILTER_VALIDATE_IP)) {
            return 'Invalid IP address';
        }
        
        $result = $api->changeIp($licenseKey, $newIp);
        
        if ($result['success']) {
            return 'IP address changed successfully';
        } else {
            return $result['message'] ?? 'Failed to change IP';
        }
    } catch (Exception $e) {
        return 'Error: ' . $e->getMessage();
    }
}

// Admin functions moved to Addon module
// Use LicenZer Addon for discount license management

function licenzer_GetLicenseKey($params)
{
    // Try to get from custom field
    $customField = Capsule::table('tblcustomfieldsvalues')
        ->join('tblcustomfields', 'tblcustomfields.id', '=', 'tblcustomfieldsvalues.fieldid')
        ->where('tblcustomfieldsvalues.relid', $params['serviceid'])
        ->where('tblcustomfields.fieldname', 'LIKE', '%License Key%')
        ->first();
    
    if ($customField) {
        return $customField->value;
    }
    
    // Try to get from service notes
    $service = Capsule::table('tblhosting')
        ->where('id', $params['serviceid'])
        ->first();
    
    if ($service && $service->notes) {
        $notes = json_decode($service->notes, true);
        return $notes['license_key'] ?? '';
    }
    
    return '';
}

/**
 * LicenZer API Class
 */
class LicenZerAPI
{
    private $apiUrl;
    private $token;
    private $softwareKey;
    
    public function __construct($params)
    {
        $this->apiUrl = rtrim($params['configoption1'] ?? '', '/');
        $this->token = $params['configoption2'] ?? '';
        $this->softwareKey = $params['configoption3'] ?? '';
    }
    
    public function registerLicense($ip, $month = 1)
    {
        $url = $this->apiUrl . '/resellerapi/register?' . http_build_query([
            'token' => $this->token,
            'key' => $this->softwareKey,
            'ip' => $ip,
            'month' => $month,
        ]);
        
        $response = $this->makeRequest($url);
        
        if (isset($response['status']) && $response['status'] == 'success') {
            return array(
                'success' => true,
                'license_key' => $response['data']['license_key'] ?? '',
                'ip' => $response['data']['ip'] ?? $ip,
                'end_at' => $response['data']['end_at'] ?? '',
                'message' => $response['message'] ?? 'License created successfully',
            );
        }
        
        return array(
            'success' => false,
            'message' => $response['message'] ?? 'Failed to register license',
        );
    }
    
    public function activateLicense($licenseKey)
    {
        // Get IP from license key
        $licenseInfo = $this->getLicenseInfo($licenseKey);
        if (empty($licenseInfo) || empty($licenseInfo['ip'])) {
            return array('success' => false, 'message' => 'License not found');
        }
        
        $url = $this->apiUrl . '/resellerapi/activate?' . http_build_query([
            'token' => $this->token,
            'key' => $this->softwareKey,
            'ip' => $licenseInfo['ip'],
        ]);
        
        $response = $this->makeRequest($url);
        
        return array(
            'success' => isset($response['status']) && $response['status'] == 'success',
            'message' => $response['message'] ?? 'Failed to activate license',
        );
    }
    
    public function deactivateLicense($licenseKey)
    {
        // Get IP from license key
        $licenseInfo = $this->getLicenseInfo($licenseKey);
        if (empty($licenseInfo) || empty($licenseInfo['ip'])) {
            return array('success' => false, 'message' => 'License not found');
        }
        
        $url = $this->apiUrl . '/resellerapi/deactivate?' . http_build_query([
            'token' => $this->token,
            'key' => $this->softwareKey,
            'ip' => $licenseInfo['ip'],
        ]);
        
        $response = $this->makeRequest($url);
        
        return array(
            'success' => isset($response['status']) && $response['status'] == 'success',
            'message' => $response['message'] ?? 'Failed to deactivate license',
        );
    }
    
    public function deleteLicense($licenseKey)
    {
        // Get IP from license key
        $licenseInfo = $this->getLicenseInfo($licenseKey);
        if (empty($licenseInfo) || empty($licenseInfo['ip'])) {
            return array('success' => false, 'message' => 'License not found');
        }
        
        $url = $this->apiUrl . '/resellerapi/delete?' . http_build_query([
            'token' => $this->token,
            'key' => $this->softwareKey,
            'ip' => $licenseInfo['ip'],
        ]);
        
        $response = $this->makeRequest($url);
        
        return array(
            'success' => isset($response['status']) && $response['status'] == 'success',
            'message' => $response['message'] ?? 'Failed to delete license',
        );
    }
    
    public function changeIp($licenseKey, $newIp)
    {
        // Get current IP from license key
        $licenseInfo = $this->getLicenseInfo($licenseKey);
        if (empty($licenseInfo) || empty($licenseInfo['ip'])) {
            return array('success' => false, 'message' => 'License not found');
        }
        
        $url = $this->apiUrl . '/resellerapi/changeiplicense?' . http_build_query([
            'token' => $this->token,
            'key' => $this->softwareKey,
            'ip' => $licenseInfo['ip'],
            'ip_new' => $newIp,
        ]);
        
        $response = $this->makeRequest($url);
        
        return array(
            'success' => isset($response['status']) && $response['status'] == 'success',
            'message' => $response['message'] ?? 'Failed to change IP',
        );
    }
    
    public function getLicenseInfo($licenseKey)
    {
        // Get license list to find IP by license_key
        $listUrl = $this->apiUrl . '/resellerapi/getlist?' . http_build_query([
            'token' => $this->token,
        ]);
        
        $listResponse = $this->makeRequest($listUrl);
        
        if (isset($listResponse['status']) && $listResponse['status'] == 'success') {
            $licenses = $listResponse['data'] ?? array();
            foreach ($licenses as $license) {
                if (isset($license['license_key']) && $license['license_key'] == $licenseKey) {
                    // Found the license, now get full info using IP and key
                    if (isset($license['ip']) && isset($license['key'])) {
                        $url = $this->apiUrl . '/resellerapi/licenseinfo?' . http_build_query([
                            'token' => $this->token,
                            'key' => $license['key'],
                            'ip' => $license['ip'],
                        ]);
                        $response = $this->makeRequest($url);
                        if (isset($response['status']) && $response['status'] == 'success') {
                            return $response['data'] ?? array();
                        }
                    }
                    // If we can't get full info, return what we have from list
                    return array(
                        'license_key' => $license['license_key'] ?? '',
                        'ip' => $license['ip'] ?? '',
                        'status' => $license['status'] ?? 0,
                        'end_at' => $license['Expired date'] ?? '',
                    );
                }
            }
        }
        
        return array();
    }
    
    public function getDiscountLicenses($softwareKey = null)
    {
        $params = array();
        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }
        
        $url = $this->apiUrl . '/resellerapi/getdiscountlicenses?' . http_build_query(array_merge([
            'token' => $this->token,
        ], $params));
        
        $response = $this->makeRequest($url);
        
        if (isset($response['status']) && $response['status'] == 'success') {
            return $response['data'] ?? array();
        }
        
        return array();
    }
    
    public function getInstallCommand($softwareKey)
    {
        // Get software info
        $url = $this->apiUrl . '/resellerapi/getsoftwares?' . http_build_query([
            'token' => $this->token,
        ]);
        
        $response = $this->makeRequest($url);
        
        if (is_array($response)) {
            foreach ($response as $software) {
                if (isset($software['key']) && $software['key'] == $softwareKey) {
                    // Get installation command from license info if available
                    return 'Installation command for ' . ($software['name'] ?? $softwareKey);
                }
            }
        }
        
        return 'Installation command for ' . $softwareKey;
    }
    
    private function makeRequest($url)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode == 200) {
            return json_decode($response, true) ?: array();
        }
        
        return array('status' => 'error', 'message' => 'API request failed');
    }
}

