<?php

/**
 * LicenZer API Helper Class
 * Provides easy access to LicenZer API functions
 */

class LicenZerAPIHelper
{
    private $apiUrl;
    private $token;
    
    public function __construct($apiUrl, $token)
    {
        $this->apiUrl = rtrim($apiUrl, '/');
        $this->token = $token;
    }
    
    public function testConnection()
    {
        $url = $this->apiUrl . '/resellerapi/getstatus?' . http_build_query([
            'token' => $this->token,
        ]);
        
        $response = $this->makeRequest($url);
        
        return array(
            'success' => isset($response['status']) && $response['status'] == 'success',
            'message' => $response['message'] ?? 'Connection failed'
        );
    }
    
    public function getStatistics()
    {
        $url = $this->apiUrl . '/resellerapi/getlist?' . http_build_query([
            'token' => $this->token,
        ]);
        
        $response = $this->makeRequest($url);
        
        if (isset($response['status']) && $response['status'] == 'success') {
            $licenses = $response['data'] ?? array();
            $total = count($licenses);
            $active = 0;
            
            foreach ($licenses as $license) {
                if (isset($license['status']) && $license['status'] == 1) {
                    $active++;
                }
            }
            
            // Get balance
            $balanceUrl = $this->apiUrl . '/resellerapi/getblanace?' . http_build_query([
                'token' => $this->token,
            ]);
            $balanceResponse = $this->makeRequest($balanceUrl);
            $balance = 0;
            if (isset($balanceResponse['status']) && $balanceResponse['status'] == 'success') {
                $balance = floatval($balanceResponse['data'] ?? 0);
            }
            
            // Get discount licenses count
            $discountUrl = $this->apiUrl . '/resellerapi/getdiscountlicenses?' . http_build_query([
                'token' => $this->token,
            ]);
            $discountResponse = $this->makeRequest($discountUrl);
            $discountCount = 0;
            if (isset($discountResponse['status']) && $discountResponse['status'] == 'success') {
                $discountCount = count($discountResponse['data'] ?? array());
            }
            
            return array(
                'total_licenses' => $total,
                'active_licenses' => $active,
                'discount_licenses' => $discountCount,
                'balance' => $balance,
            );
        }
        
        return array(
            'total_licenses' => 0,
            'active_licenses' => 0,
            'discount_licenses' => 0,
            'balance' => 0,
        );
    }
    
    public function getAllLicenses()
    {
        $url = $this->apiUrl . '/resellerapi/getlist?' . http_build_query([
            'token' => $this->token,
        ]);
        
        $response = $this->makeRequest($url);
        
        if (isset($response['status']) && $response['status'] == 'success') {
            return $response['data'] ?? array();
        }
        
        return array();
    }
    
    public function getRecentLicenses($limit = 10)
    {
        $licenses = $this->getAllLicenses();
        return array_slice($licenses, 0, $limit);
    }
    
    public function getDiscountLicenses($softwareKey = null)
    {
        $params = array('token' => $this->token);
        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }
        
        $url = $this->apiUrl . '/resellerapi/getdiscountlicenses?' . http_build_query($params);
        
        $response = $this->makeRequest($url);
        
        if (isset($response['status']) && $response['status'] == 'success') {
            return $response['data'] ?? array();
        }
        
        return array();
    }
    
    public function changeIp($licenseKey, $newIp)
    {
        // Get current IP from license list
        $licenses = $this->getAllLicenses();
        $currentIp = '';
        $softwareKey = '';
        
        foreach ($licenses as $license) {
            if (isset($license['license_key']) && $license['license_key'] == $licenseKey) {
                $currentIp = $license['ip'] ?? '';
                $softwareKey = $license['key'] ?? '';
                break;
            }
        }
        
        if (empty($currentIp) || empty($softwareKey)) {
            return array('success' => false, 'message' => 'License not found');
        }
        
        $url = $this->apiUrl . '/resellerapi/changeiplicense?' . http_build_query([
            'token' => $this->token,
            'key' => $softwareKey,
            'ip' => $currentIp,
            'ip_new' => $newIp,
        ]);
        
        $response = $this->makeRequest($url);
        
        return array(
            'success' => isset($response['status']) && $response['status'] == 'success',
            'message' => $response['message'] ?? 'Failed to change IP'
        );
    }
    
    public function getLicenseInfo($licenseKey)
    {
        // Get license list to find IP by license_key
        $licenses = $this->getAllLicenses();
        
        foreach ($licenses as $license) {
            if (isset($license['license_key']) && $license['license_key'] == $licenseKey) {
                // Found the license, now get full info using IP and key
                if (isset($license['ip']) && isset($license['key'])) {
                    $url = $this->apiUrl . '/resellerapi/licenseinfo?' . http_build_query([
                        'token' => $this->token,
                        'key' => $license['key'],
                        'ip' => $license['ip'],
                    ]);
                    $response = $this->makeRequest($url);
                    if (isset($response['status']) && $response['status'] == 'success') {
                        return $response['data'] ?? array();
                    }
                }
                // If we can't get full info, return what we have from list
                return array(
                    'license_key' => $license['license_key'] ?? '',
                    'ip' => $license['ip'] ?? '',
                    'status' => $license['status'] ?? 0,
                    'end_at' => $license['end_at'] ?? $license['Expired date'] ?? '',
                );
            }
        }
        
        return array();
    }
    
    public function assignDiscountLicense($licenseKey, $serviceId, $newIp = '')
    {
        // Get license info
        $licenses = $this->getAllLicenses();
        $licenseInfo = null;
        
        foreach ($licenses as $license) {
            if (isset($license['license_key']) && $license['license_key'] == $licenseKey) {
                $licenseInfo = $license;
                break;
            }
        }
        
        if (!$licenseInfo) {
            return array('success' => false, 'message' => 'License not found');
        }
        
        // Change IP if needed
        if (!empty($newIp) && filter_var($newIp, FILTER_VALIDATE_IP)) {
            $changeResult = $this->changeIp($licenseKey, $newIp);
            if (!$changeResult['success']) {
                return $changeResult;
            }
        }
        
        // Update WHMCS service
        use WHMCS\Database\Capsule;
        
        $service = Capsule::table('tblhosting')->where('id', $serviceId)->first();
        if (!$service) {
            return array('success' => false, 'message' => 'Service not found');
        }
        
        // Update service notes
        $notes = json_decode($service->notes ?? '{}', true);
        $notes['license_key'] = $licenseKey;
        $notes['ip_changes'] = 0;
        $notes['is_discount_license'] = true;
        
        Capsule::table('tblhosting')
            ->where('id', $serviceId)
            ->update(['notes' => json_encode($notes)]);
        
        // Update custom field if exists
        $customField = Capsule::table('tblcustomfields')
            ->where('type', 'product')
            ->where('relid', $service->packageid)
            ->where('fieldname', 'LIKE', '%License Key%')
            ->first();
        
        if ($customField) {
            Capsule::table('tblcustomfieldsvalues')
                ->updateOrInsert(
                    ['fieldid' => $customField->id, 'relid' => $serviceId],
                    ['value' => $licenseKey]
                );
        }
        
        return array('success' => true, 'message' => 'Discount license assigned successfully');
    }
    
    private function makeRequest($url)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode == 200) {
            return json_decode($response, true) ?: array();
        }
        
        return array('status' => 'error', 'message' => 'API request failed');
    }
}

