<?php
/**
 * LicenZer Web Example
 * 
 * Example web interface for LicenZer API
 */

require_once __DIR__ . '/licenzer.php';

// Load config
$config = [];
if (file_exists(__DIR__ . '/config.php')) {
    $config = require __DIR__ . '/config.php';
} else {
    $config = [
        'api_url' => LICENZER_API_URL,
        'reseller_token' => LICENZER_TOKEN,
    ];
}

$api = new LicenzerAPI($config['api_url'], $config['reseller_token']);

// Handle AJAX requests
if (isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'get_licenses':
            $result = $api->getLicenses();
            echo json_encode($result);
            exit;
            
        case 'register_license':
            $ip = $_POST['ip'] ?? '';
            $month = (int)($_POST['month'] ?? 1);
            $softwareKey = $_POST['software_key'] ?? null;
            $result = $api->registerLicense($ip, $month, $softwareKey);
            echo json_encode($result);
            exit;
            
        case 'change_ip':
            $ip = $_POST['ip'] ?? '';
            $newIp = $_POST['new_ip'] ?? '';
            $softwareKey = $_POST['software_key'] ?? '';
            $result = $api->changeIp($ip, $newIp, $softwareKey);
            echo json_encode($result);
            exit;
            
        case 'get_stats':
            $licenses = $api->getLicenses();
            $balance = $api->getBalance();
            $discountLicenses = $api->getDiscountLicenses();
            
            $stats = [
                'total' => 0,
                'active' => 0,
                'inactive' => 0,
                'discount' => 0,
                'balance' => 0,
            ];
            
            if (isset($licenses['status']) && $licenses['status'] == 'success') {
                $licenseList = $licenses['data'] ?? [];
                $stats['total'] = count($licenseList);
                foreach ($licenseList as $license) {
                    if (isset($license['status']) && $license['status'] == 1) {
                        $stats['active']++;
                    } else {
                        $stats['inactive']++;
                    }
                }
            }
            
            if (isset($balance['status']) && $balance['status'] == 'success') {
                $stats['balance'] = floatval($balance['data'] ?? 0);
            }
            
            if (isset($discountLicenses['status']) && $discountLicenses['status'] == 'success') {
                $stats['discount'] = count($discountLicenses['data'] ?? []);
            }
            
            echo json_encode(['status' => 'success', 'data' => $stats]);
            exit;
    }
}

// Get initial data
$licenses = $api->getLicenses();
$licenseList = [];
if (isset($licenses['status']) && $licenses['status'] == 'success') {
    $licenseList = $licenses['data'] ?? [];
}

$balance = $api->getBalance();
$balanceAmount = 0;
if (isset($balance['status']) && $balance['status'] == 'success') {
    $balanceAmount = floatval($balance['data'] ?? 0);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>LicenZer Management</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .stat-card {
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            color: white;
        }
        .stat-card.primary { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .stat-card.success { background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%); }
        .stat-card.warning { background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); }
        .stat-card.info { background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); }
    </style>
</head>
<body>
    <nav class="navbar navbar-dark bg-dark">
        <div class="container-fluid">
            <span class="navbar-brand mb-0 h1">LicenZer Management</span>
        </div>
    </nav>

    <div class="container mt-4">
        <!-- Statistics -->
        <div class="row mb-4" id="stats-container">
            <div class="col-md-3">
                <div class="stat-card primary">
                    <h5>Total Licenses</h5>
                    <h2 id="stat-total">-</h2>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card success">
                    <h5>Active Licenses</h5>
                    <h2 id="stat-active">-</h2>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card warning">
                    <h5>Discount Licenses</h5>
                    <h2 id="stat-discount">-</h2>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card info">
                    <h5>Balance</h5>
                    <h2 id="stat-balance">-</h2>
                </div>
            </div>
        </div>

        <!-- Licenses Table -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">All Licenses</h5>
                <button class="btn btn-primary btn-sm" onclick="refreshData()">
                    <i class="fas fa-sync-alt"></i> Refresh
                </button>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>License Key</th>
                                <th>IP Address</th>
                                <th>Software</th>
                                <th>Status</th>
                                <th>Expiry Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="licenses-table">
                            <?php foreach ($licenseList as $license): ?>
                            <tr>
                                <td><code><?php echo htmlspecialchars($license['license_key'] ?? 'N/A'); ?></code></td>
                                <td><?php echo htmlspecialchars($license['ip'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($license['software_name'] ?? $license['key'] ?? 'N/A'); ?></td>
                                <td>
                                    <?php if ($license['status'] ?? 0): ?>
                                        <span class="badge bg-success">Active</span>
                                    <?php else: ?>
                                        <span class="badge bg-danger">Inactive</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars($license['end_at'] ?? $license['Expired date'] ?? 'N/A'); ?></td>
                                <td>
                                    <button class="btn btn-warning btn-sm" onclick="showChangeIpModal('<?php echo htmlspecialchars($license['ip'] ?? ''); ?>', '<?php echo htmlspecialchars($license['key'] ?? ''); ?>')">
                                        <i class="fas fa-exchange-alt"></i> Change IP
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Change IP Modal -->
    <div class="modal fade" id="changeIpModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form id="change-ip-form">
                    <div class="modal-header">
                        <h5 class="modal-title">Change IP Address</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <input type="hidden" name="action" value="change_ip">
                        <input type="hidden" name="ip" id="modal-ip">
                        <input type="hidden" name="software_key" id="modal-software-key">
                        
                        <div class="mb-3">
                            <label class="form-label">Current IP</label>
                            <input type="text" id="modal-current-ip-display" class="form-control" readonly>
                        </div>
                        <div class="mb-3">
                            <label for="new-ip" class="form-label">New IP Address</label>
                            <input type="text" name="new_ip" id="new-ip" class="form-control" required pattern="^([0-9]{1,3}\.){3}[0-9]{1,3}$" placeholder="192.168.1.1">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Change IP</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function refreshData() {
            // Refresh stats
            fetch('', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=get_stats'
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    document.getElementById('stat-total').textContent = data.data.total;
                    document.getElementById('stat-active').textContent = data.data.active;
                    document.getElementById('stat-discount').textContent = data.data.discount;
                    document.getElementById('stat-balance').textContent = '$' + data.data.balance.toFixed(2);
                }
            });

            // Refresh licenses
            fetch('', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=get_licenses'
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    const tbody = document.getElementById('licenses-table');
                    tbody.innerHTML = '';
                    data.data.forEach(license => {
                        const row = document.createElement('tr');
                        row.innerHTML = `
                            <td><code>${license.license_key || 'N/A'}</code></td>
                            <td>${license.ip || 'N/A'}</td>
                            <td>${license.software_name || license.key || 'N/A'}</td>
                            <td><span class="badge bg-${license.status ? 'success' : 'danger'}">${license.status ? 'Active' : 'Inactive'}</span></td>
                            <td>${license.end_at || license['Expired date'] || 'N/A'}</td>
                            <td>
                                <button class="btn btn-warning btn-sm" onclick="showChangeIpModal('${license.ip || ''}', '${license.key || ''}')">
                                    <i class="fas fa-exchange-alt"></i> Change IP
                                </button>
                            </td>
                        `;
                        tbody.appendChild(row);
                    });
                }
            });
        }

        function showChangeIpModal(ip, softwareKey) {
            document.getElementById('modal-ip').value = ip;
            document.getElementById('modal-software-key').value = softwareKey;
            document.getElementById('modal-current-ip-display').value = ip;
            const modal = new bootstrap.Modal(document.getElementById('changeIpModal'));
            modal.show();
        }

        document.getElementById('change-ip-form').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    alert('IP changed successfully!');
                    bootstrap.Modal.getInstance(document.getElementById('changeIpModal')).hide();
                    refreshData();
                } else {
                    alert('Error: ' + (data.message || 'Unknown error'));
                }
            });
        });

        // Load stats on page load
        refreshData();
    </script>
</body>
</html>

