<?php
/**
 * LicenZer PHP Script
 * 
 * A standalone PHP script for managing LicenZer licenses
 * 
 * @version 1.0.0
 * @author LicenZer
 * @license MIT
 */

class LicenzerAPI {
    private $apiUrl;
    private $token;
    
    public function __construct($apiUrl, $token) {
        $this->apiUrl = rtrim($apiUrl, '/');
        $this->token = $token;
    }
    
    /**
     * Make API request
     */
    private function request($endpoint, $params = []) {
        $params['token'] = $this->token;
        $url = $this->apiUrl . '/' . ltrim($endpoint, '/') . '?' . http_build_query($params);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['status' => 'error', 'message' => $error];
        }
        
        if ($httpCode == 200) {
            $data = json_decode($response, true);
            return $data ?: ['status' => 'error', 'message' => 'Invalid response'];
        }
        
        return ['status' => 'error', 'message' => 'HTTP Error: ' . $httpCode];
    }
    
    /**
     * Register new license
     */
    public function registerLicense($ip, $month = 1, $softwareKey = null) {
        $params = [
            'ip' => $ip,
            'month' => $month,
        ];
        
        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }
        
        return $this->request('/resellerapi/register', $params);
    }
    
    /**
     * Activate license
     */
    public function activateLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/activate', [
            'ip' => $ip,
            'key' => $softwareKey,
        ]);
    }
    
    /**
     * Deactivate license
     */
    public function deactivateLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/deactivate', [
            'ip' => $ip,
            'key' => $softwareKey,
        ]);
    }
    
    /**
     * Delete license
     */
    public function deleteLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/delete', [
            'ip' => $ip,
            'key' => $softwareKey,
        ]);
    }
    
    /**
     * Change license IP
     */
    public function changeIp($ip, $newIp, $softwareKey) {
        return $this->request('/resellerapi/changeiplicense', [
            'ip' => $ip,
            'ip_new' => $newIp,
            'key' => $softwareKey,
        ]);
    }
    
    /**
     * Get license info
     */
    public function getLicenseInfo($ip, $softwareKey) {
        return $this->request('/resellerapi/licenseinfo', [
            'ip' => $ip,
            'key' => $softwareKey,
        ]);
    }
    
    /**
     * Get all licenses
     */
    public function getLicenses() {
        return $this->request('/resellerapi/getlist');
    }
    
    /**
     * Get discount licenses
     */
    public function getDiscountLicenses($softwareKey = null) {
        $params = [];
        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }
        return $this->request('/resellerapi/getdiscountlicenses', $params);
    }
    
    /**
     * Get available softwares
     */
    public function getSoftwares() {
        return $this->request('/resellerapi/getsoftwares');
    }
    
    /**
     * Get reseller balance
     */
    public function getBalance() {
        return $this->request('/resellerapi/getblanace');
    }
    
    /**
     * Test connection
     */
    public function testConnection() {
        $result = $this->request('/resellerapi/getstatus');
        return isset($result['status']) && $result['status'] == 'success';
    }
}

/**
 * Configuration
 */
define('LICENZER_API_URL', 'https://licenzer.pro');
define('LICENZER_TOKEN', 'YOUR_RESELLER_TOKEN_HERE');

/**
 * Initialize API
 */
$api = new LicenzerAPI(LICENZER_API_URL, LICENZER_TOKEN);

/**
 * Example Usage
 */

// Test connection
echo "Testing connection...\n";
if ($api->testConnection()) {
    echo "✓ Connection successful!\n\n";
} else {
    echo "✗ Connection failed! Please check your API URL and token.\n\n";
    exit(1);
}

// Get statistics
echo "=== Statistics ===\n";
$licenses = $api->getLicenses();
if (isset($licenses['status']) && $licenses['status'] == 'success') {
    $licenseList = $licenses['data'] ?? [];
    $total = count($licenseList);
    $active = 0;
    foreach ($licenseList as $license) {
        if (isset($license['status']) && $license['status'] == 1) {
            $active++;
        }
    }
    echo "Total Licenses: {$total}\n";
    echo "Active Licenses: {$active}\n";
    echo "Inactive Licenses: " . ($total - $active) . "\n";
}

$balance = $api->getBalance();
if (isset($balance['status']) && $balance['status'] == 'success') {
    echo "Balance: $" . number_format($balance['data'] ?? 0, 2) . "\n";
}

$discountLicenses = $api->getDiscountLicenses();
if (isset($discountLicenses['status']) && $discountLicenses['status'] == 'success') {
    echo "Discount Licenses: " . count($discountLicenses['data'] ?? []) . "\n";
}

echo "\n";

// Get all licenses
echo "=== All Licenses ===\n";
if (isset($licenses['status']) && $licenses['status'] == 'success') {
    foreach ($licenseList as $license) {
        echo "License Key: " . ($license['license_key'] ?? 'N/A') . "\n";
        echo "IP: " . ($license['ip'] ?? 'N/A') . "\n";
        echo "Software: " . ($license['software_name'] ?? $license['key'] ?? 'N/A') . "\n";
        echo "Status: " . (($license['status'] ?? 0) ? 'Active' : 'Inactive') . "\n";
        echo "Expiry: " . ($license['end_at'] ?? $license['Expired date'] ?? 'N/A') . "\n";
        echo "---\n";
    }
} else {
    echo "No licenses found or error: " . ($licenses['message'] ?? 'Unknown error') . "\n";
}

echo "\n";

// Example: Register a new license
/*
echo "=== Register New License ===\n";
$result = $api->registerLicense('192.168.1.100', 1, 'software_key');
if (isset($result['status']) && $result['status'] == 'success') {
    echo "License registered successfully!\n";
    echo "License Key: " . ($result['data']['license_key'] ?? 'N/A') . "\n";
} else {
    echo "Error: " . ($result['message'] ?? 'Unknown error') . "\n";
}
echo "\n";
*/

// Example: Change IP
/*
echo "=== Change IP ===\n";
$result = $api->changeIp('192.168.1.100', '192.168.1.101', 'software_key');
if (isset($result['status']) && $result['status'] == 'success') {
    echo "IP changed successfully!\n";
} else {
    echo "Error: " . ($result['message'] ?? 'Unknown error') . "\n";
}
echo "\n";
*/

// Example: Get license info
/*
echo "=== Get License Info ===\n";
$result = $api->getLicenseInfo('192.168.1.100', 'software_key');
if (isset($result['status']) && $result['status'] == 'success') {
    $data = $result['data'] ?? [];
    echo "License Key: " . ($data['license_key'] ?? 'N/A') . "\n";
    echo "IP: " . ($data['ip'] ?? 'N/A') . "\n";
    echo "Status: " . (($data['status'] ?? 0) ? 'Active' : 'Inactive') . "\n";
    echo "Expiry: " . ($data['end_at'] ?? 'N/A') . "\n";
} else {
    echo "Error: " . ($result['message'] ?? 'Unknown error') . "\n";
}
echo "\n";
*/

// Example: Get discount licenses
/*
echo "=== Discount Licenses ===\n";
$discountLicenses = $api->getDiscountLicenses();
if (isset($discountLicenses['status']) && $discountLicenses['status'] == 'success') {
    foreach ($discountLicenses['data'] ?? [] as $license) {
        echo "License Key: " . ($license['license_key'] ?? 'N/A') . "\n";
        echo "Software: " . ($license['software_name'] ?? 'N/A') . "\n";
        echo "Original Price: $" . number_format($license['original_price'] ?? 0, 2) . "\n";
        echo "Discount Price: $" . number_format($license['price'] ?? 0, 2) . "\n";
        echo "---\n";
    }
} else {
    echo "No discount licenses found.\n";
}
echo "\n";
*/

?>

