<?php
/**
 * LicenZer CLI Script
 * 
 * Command-line interface for LicenZer API
 * 
 * Usage:
 *   php licenzer-cli.php list
 *   php licenzer-cli.php register <ip> <month> [software_key]
 *   php licenzer-cli.php change-ip <ip> <new_ip> <software_key>
 *   php licenzer-cli.php info <ip> <software_key>
 *   php licenzer-cli.php balance
 *   php licenzer-cli.php discount
 */

require_once __DIR__ . '/licenzer.php';

// Load config if exists
$config = [];
if (file_exists(__DIR__ . '/config.php')) {
    $config = require __DIR__ . '/config.php';
} else {
    $config = [
        'api_url' => LICENZER_API_URL,
        'reseller_token' => LICENZER_TOKEN,
    ];
}

$api = new LicenzerAPI($config['api_url'], $config['reseller_token']);

// Get command
$command = $argv[1] ?? null;

if (!$command) {
    echo "LicenZer CLI\n";
    echo "Usage: php licenzer-cli.php <command> [arguments]\n\n";
    echo "Commands:\n";
    echo "  list                    - List all licenses\n";
    echo "  register <ip> <month> [key] - Register new license\n";
    echo "  change-ip <ip> <new_ip> <key> - Change license IP\n";
    echo "  info <ip> <key>         - Get license info\n";
    echo "  balance                 - Get reseller balance\n";
    echo "  discount                - Get discount licenses\n";
    echo "  stats                   - Get statistics\n";
    echo "  test                    - Test connection\n";
    exit(0);
}

switch ($command) {
    case 'test':
        echo "Testing connection...\n";
        if ($api->testConnection()) {
            echo "✓ Connection successful!\n";
        } else {
            echo "✗ Connection failed!\n";
            exit(1);
        }
        break;
        
    case 'list':
        echo "Fetching licenses...\n\n";
        $licenses = $api->getLicenses();
        if (isset($licenses['status']) && $licenses['status'] == 'success') {
            $licenseList = $licenses['data'] ?? [];
            if (empty($licenseList)) {
                echo "No licenses found.\n";
            } else {
                foreach ($licenseList as $license) {
                    echo "License Key: " . ($license['license_key'] ?? 'N/A') . "\n";
                    echo "IP: " . ($license['ip'] ?? 'N/A') . "\n";
                    echo "Software: " . ($license['software_name'] ?? $license['key'] ?? 'N/A') . "\n";
                    echo "Status: " . (($license['status'] ?? 0) ? 'Active' : 'Inactive') . "\n";
                    echo "Expiry: " . ($license['end_at'] ?? $license['Expired date'] ?? 'N/A') . "\n";
                    echo "---\n";
                }
            }
        } else {
            echo "Error: " . ($licenses['message'] ?? 'Unknown error') . "\n";
            exit(1);
        }
        break;
        
    case 'register':
        if (!isset($argv[2]) || !isset($argv[3])) {
            echo "Error: IP and month are required\n";
            echo "Usage: php licenzer-cli.php register <ip> <month> [software_key]\n";
            exit(1);
        }
        $ip = $argv[2];
        $month = (int)$argv[3];
        $softwareKey = $argv[4] ?? null;
        
        echo "Registering license...\n";
        $result = $api->registerLicense($ip, $month, $softwareKey);
        if (isset($result['status']) && $result['status'] == 'success') {
            echo "✓ License registered successfully!\n";
            echo "License Key: " . ($result['data']['license_key'] ?? 'N/A') . "\n";
            echo "IP: " . ($result['data']['ip'] ?? $ip) . "\n";
            echo "Expiry: " . ($result['data']['end_at'] ?? 'N/A') . "\n";
        } else {
            echo "✗ Error: " . ($result['message'] ?? 'Unknown error') . "\n";
            exit(1);
        }
        break;
        
    case 'change-ip':
        if (!isset($argv[2]) || !isset($argv[3]) || !isset($argv[4])) {
            echo "Error: IP, new IP, and software key are required\n";
            echo "Usage: php licenzer-cli.php change-ip <ip> <new_ip> <software_key>\n";
            exit(1);
        }
        $ip = $argv[2];
        $newIp = $argv[3];
        $softwareKey = $argv[4];
        
        echo "Changing IP...\n";
        $result = $api->changeIp($ip, $newIp, $softwareKey);
        if (isset($result['status']) && $result['status'] == 'success') {
            echo "✓ IP changed successfully!\n";
        } else {
            echo "✗ Error: " . ($result['message'] ?? 'Unknown error') . "\n";
            exit(1);
        }
        break;
        
    case 'info':
        if (!isset($argv[2]) || !isset($argv[3])) {
            echo "Error: IP and software key are required\n";
            echo "Usage: php licenzer-cli.php info <ip> <software_key>\n";
            exit(1);
        }
        $ip = $argv[2];
        $softwareKey = $argv[3];
        
        echo "Fetching license info...\n\n";
        $result = $api->getLicenseInfo($ip, $softwareKey);
        if (isset($result['status']) && $result['status'] == 'success') {
            $data = $result['data'] ?? [];
            echo "License Key: " . ($data['license_key'] ?? 'N/A') . "\n";
            echo "IP: " . ($data['ip'] ?? $ip) . "\n";
            echo "Software: " . ($data['software_key'] ?? $softwareKey) . "\n";
            echo "Status: " . (($data['status'] ?? 0) ? 'Active' : 'Inactive') . "\n";
            echo "Expiry: " . ($data['end_at'] ?? 'N/A') . "\n";
            echo "Change IP: " . ($data['change_ip'] ?? 'N/A') . "\n";
        } else {
            echo "✗ Error: " . ($result['message'] ?? 'Unknown error') . "\n";
            exit(1);
        }
        break;
        
    case 'balance':
        echo "Fetching balance...\n";
        $balance = $api->getBalance();
        if (isset($balance['status']) && $balance['status'] == 'success') {
            echo "Balance: $" . number_format($balance['data'] ?? 0, 2) . "\n";
        } else {
            echo "✗ Error: " . ($balance['message'] ?? 'Unknown error') . "\n";
            exit(1);
        }
        break;
        
    case 'discount':
        echo "Fetching discount licenses...\n\n";
        $discountLicenses = $api->getDiscountLicenses();
        if (isset($discountLicenses['status']) && $discountLicenses['status'] == 'success') {
            $licenseList = $discountLicenses['data'] ?? [];
            if (empty($licenseList)) {
                echo "No discount licenses available.\n";
            } else {
                foreach ($licenseList as $license) {
                    echo "License Key: " . ($license['license_key'] ?? 'N/A') . "\n";
                    echo "Software: " . ($license['software_name'] ?? 'N/A') . "\n";
                    echo "IP: " . ($license['ip'] ?? 'N/A') . "\n";
                    echo "Original Price: $" . number_format($license['original_price'] ?? 0, 2) . "\n";
                    echo "Discount Price: $" . number_format($license['price'] ?? 0, 2) . "\n";
                    echo "Expiry: " . ($license['expiry_date'] ?? 'N/A') . "\n";
                    echo "---\n";
                }
            }
        } else {
            echo "✗ Error: " . ($discountLicenses['message'] ?? 'Unknown error') . "\n";
            exit(1);
        }
        break;
        
    case 'stats':
        echo "Fetching statistics...\n\n";
        $licenses = $api->getLicenses();
        $balance = $api->getBalance();
        $discountLicenses = $api->getDiscountLicenses();
        
        $stats = [
            'total' => 0,
            'active' => 0,
            'inactive' => 0,
            'discount' => 0,
            'balance' => 0,
        ];
        
        if (isset($licenses['status']) && $licenses['status'] == 'success') {
            $licenseList = $licenses['data'] ?? [];
            $stats['total'] = count($licenseList);
            foreach ($licenseList as $license) {
                if (isset($license['status']) && $license['status'] == 1) {
                    $stats['active']++;
                } else {
                    $stats['inactive']++;
                }
            }
        }
        
        if (isset($balance['status']) && $balance['status'] == 'success') {
            $stats['balance'] = floatval($balance['data'] ?? 0);
        }
        
        if (isset($discountLicenses['status']) && $discountLicenses['status'] == 'success') {
            $stats['discount'] = count($discountLicenses['data'] ?? []);
        }
        
        echo "Total Licenses: {$stats['total']}\n";
        echo "Active: {$stats['active']}\n";
        echo "Inactive: {$stats['inactive']}\n";
        echo "Discount Licenses: {$stats['discount']}\n";
        echo "Balance: $" . number_format($stats['balance'], 2) . "\n";
        break;
        
    default:
        echo "Unknown command: {$command}\n";
        echo "Run 'php licenzer-cli.php' for help.\n";
        exit(1);
}

?>

