<?php
/**
 * LicenZer API Library
 */
class LicenzerApi {
    private $apiUrl;
    private $token;

    public function __construct($apiUrl, $token) {
        $this->apiUrl = rtrim($apiUrl, '/');
        $this->token = $token;
    }

    /**
     * Make API request
     */
    private function request($endpoint, $params = array()) {
        $params['token'] = $this->token;
        $url = $this->apiUrl . '/' . ltrim($endpoint, '/') . '?' . http_build_query($params);

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode == 200) {
            $data = json_decode($response, true);
            return $data ?: array('status' => 'error', 'message' => 'Invalid response');
        }
        
        return array('status' => 'error', 'message' => 'HTTP Error: ' . $httpCode);
    }

    /**
     * Register new license
     */
    public function registerLicense($ip, $month = 1, $softwareKey = null) {
        $params = array(
            'ip' => $ip,
            'month' => $month,
        );

        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }

        return $this->request('/resellerapi/register', $params);
    }

    /**
     * Activate license
     */
    public function activateLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/activate', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Deactivate license
     */
    public function deactivateLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/deactivate', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Delete license
     */
    public function deleteLicense($ip, $softwareKey) {
        return $this->request('/resellerapi/delete', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Change license IP
     */
    public function changeIp($ip, $newIp, $softwareKey) {
        return $this->request('/resellerapi/changeiplicense', array(
            'ip' => $ip,
            'ip_new' => $newIp,
            'key' => $softwareKey,
        ));
    }

    /**
     * Get license info
     */
    public function getLicenseInfo($ip, $softwareKey) {
        return $this->request('/resellerapi/licenseinfo', array(
            'ip' => $ip,
            'key' => $softwareKey,
        ));
    }

    /**
     * Get all licenses
     */
    public function getLicenses() {
        return $this->request('/resellerapi/getlist');
    }

    /**
     * Get discount licenses
     */
    public function getDiscountLicenses($softwareKey = null) {
        $params = array();
        if ($softwareKey) {
            $params['key'] = $softwareKey;
        }
        return $this->request('/resellerapi/getdiscountlicenses', $params);
    }

    /**
     * Get available softwares
     */
    public function getSoftwares() {
        return $this->request('/resellerapi/getsoftwares');
    }

    /**
     * Get reseller balance
     */
    public function getBalance() {
        return $this->request('/resellerapi/getblanace');
    }

    /**
     * Test connection
     */
    public function testConnection() {
        $result = $this->request('/resellerapi/getstatus');
        return isset($result['status']) && $result['status'] == 'success';
    }
}

